package mcfall.raytracer;

import mcfall.math.ColumnVector;
import mcfall.math.IncompatibleMatrixException;
import mcfall.math.Point;
import mcfall.math.Vector;
import mcfall.raytracer.objects.RGBIntensity;

import org.apache.log4j.Logger;

/**
 * LightSource represents a light in the scene.  Lights are described by their intensity values (which range from 0 to 1
 * for each of the three light colors of red, green and blue), and their position in space.  Different light sources
 * interact with objects differently - this class represents a light source that emanates light in all directions equally.
 * Subclasses, such as the SpotlightSource, implement specialized lights
 * @author mcfall
 *
 */
public class LightSource {
	protected Intensity red;
	protected Intensity blue;
	protected Intensity green;
	
	private Point location;
	private static Logger logger = Logger.getLogger(LightSource.class);
	
	/**
	 * Constructs a LightSource with the given color intensities and the given Point location
	 * @param red the intensity of the red component of the light source, ranging from 0 to 1
	 * @param green the intensity of the green component of the light source, ranging from 0 to 1
	 * @param blue the intensity of the blue component of the light source, ranging from 0 to 1
	 * @param location the point in space where the light source is located
	 */
	public LightSource (double red, double green, double blue, Point location) {
		this (red, green, blue, location.getX(), location.getY(), location.getZ());
	}
	
	/**
	 * Constructs a LightSource with the given color intensities and the given (x, y, z) location in space
	 * @param red the intensity of the red component of the light source, ranging from 0 to 1
	 * @param green the intensity of the green component of the light source, ranging from 0 to 1
	 * @param blue the intensity of the blue component of the light source, ranging from 0 to 1
	 * @param x the x-coordinate of the light source's location
	 * @param y the y-coordinate of the light source's location
	 * @param z the z-coordinate of the light source's location
	 */
	public LightSource (double red, double green, double blue, double x, double y, double z) {
		this.red = new Intensity (red);
		this.green = new Intensity (green);
		this.blue = new Intensity (blue);
		location = new Point (x, y, z);
	}

	/**
	 * Constructs a white light source located at the specified location
	 * with the specified intensity 
	 * @param location the location of the light source in space
	 * @param intensity the intensity of the white source; values will be
	 * clamped to the range 0.0 to 1.0
	 */
	public LightSource (double intensity, Point location) {
		this (intensity, intensity, intensity, location);
	}
	
	/**
	 * This method computes the intensity of this light's contribution due to diffuse scattering of light
	 * @param objectLocation the point the light is striking
	 * @param objectNormal the normal vector pointing <b>away</b> from the surface at the point <i>objectLocation</i>
	 * @return the intensity of the diffuse contribution for all three color components, 
	 * with each clamped to range from 0 to 1.  This is calculated by multiplying 
	 * the intensity of the light by computing the angle between the normal vector and a ray from the point in question to
	 * the light course; if this number is negative, then we clamp it to zero
	 */
	public RGBIntensity computeDiffuse(Point objectLocation, Vector objectNormal) {
		Vector s = new ColumnVector(objectLocation, location);

		//  To avoid object creation, don't create normalized versions of these vectors

		Intensity diffuseIntensity = new Intensity (s.dot(objectNormal)/(s.length()*objectNormal.length()));
		double diffuse = diffuseIntensity.getValue();
		return new RGBIntensity (
				red.getValue()*diffuse, 
				green.getValue()*diffuse, 
				blue.getValue()*diffuse
		);		
	}
	
	/**
	 * This method computes the intensity of this light's contribution due to specular reflection.
	 * Specular reflection depends on the angle between the direction of perfect reflection of the
	 * ray from the point to the light source, and the ray from the point to the eye  
	 * @param eyeLocation the location of the camera 
	 * @param objectLocation the point the light is striking
	 * @param objectNormal the normal vector pointing <b>away</b> from the surface at the point <i>objectLocation</i>
	 * @param falloff a number that denotes how mirrorlike the reflection is, typical values range
	 * from 1 (less mirrorlike) to 200 (very mirrorlike)
	 * @return the intensity of the specular contribution for all three color components, 
	 * with each clamped to range from 0 to 1
	 * 
	 */
	public RGBIntensity computeSpecular(Point cameraLocation, Point objectLocation, Vector objectNormal, int falloff) {					
		Vector v = new ColumnVector (objectLocation, cameraLocation);			
		Vector s = new ColumnVector (objectLocation, location);
		Vector r = s.reflect(objectNormal);

		//  Finally, compute the intensity of the specular components,
		//  and return it
		double intensity = Math.pow(r.dot(v)/(r.length()*v.length()), falloff);
		return new RGBIntensity (
				red.getValue()*intensity,
				green.getValue()*intensity,
				blue.getValue()*intensity
		);
	}
	
	/**
	 * Determines whether or not this light is subject to shadowing
	 * @return true if objects can block this LightSource's rays, false otherwise
	 */
	public boolean isSubjectToShadowing () {
		return true;
	}
	
	public String toString () {
		return "Light located at " + location + " with intensities (" + red + ", " + green + ", " + blue + ")";
	}
	
	protected double getBlue() {
		return blue.getValue();
	}

	protected void setBlue(double blue) {
		this.blue.setValue(blue);
	}

	protected double getGreen() {
		return green.getValue();
	}

	protected void setGreen(double green) {
		this.green.setValue(green);
	}

	public Point getLocation() {
		return location;
	}

	protected void setLocation(Point location) {
		this.location = location;
	}

	protected double getRed() {
		return red.getValue();
	}

	protected void setRed(double red) {
		this.red.setValue(red);
	}
}
